// =============================================================================
//  ZAP Audio Codec v1.0.0 (Release / English)
// =============================================================================
//  Core Engine: Studio Reference Quality
//  - Smart Stereo • Ambience Preservation • Transparent Dynamics
// =============================================================================

use std::env;
use std::f64::consts::PI;
use std::fs::File;
use std::io::Write;
use std::path::Path;
use std::sync::atomic::{AtomicBool, Ordering};
use std::sync::{Arc, Mutex};
use std::time::{Duration, Instant};

// --- External Crates ---
use cpal::traits::{DeviceTrait, HostTrait, StreamTrait};
use lofty::{Accessor, Probe, TaggedFileExt};
use rayon::prelude::*;
use rustdct::{Dct4, DctPlanner};
use serde::{Deserialize, Serialize};
use symphonia::core::io::MediaSourceStream;
use symphonia::core::probe::Hint;

// --- ENGINE SETTINGS ---

/// Base scaling factor for quantization precision.
const PRECISION_BASE: f32 = 640_000.0;

/// FFT/DCT block size. Determines frequency resolution.
const BLOCK_SIZE: usize = 2048;

/// Exponent for the power-law quantization (psychoacoustic companding).
const POWER_LAW_EXPONENT: f32 = 0.85;
const INV_POWER_LAW: f32 = 1.0 / POWER_LAW_EXPONENT;

/// Threshold below which signals are treated as zero during linear processing.
const LINEAR_DEADZONE: f32 = 0.00002;

/// Maximum target bitrate in kbps for the optimization loop.
const MAX_ALLOWED_KBPS: u32 = 280;

/// File signature (Magic Bytes) to identify ZAP files.
const ZAP_MAGIC: [u8; 4] = [0x5A, 0x41, 0x50, 0x31]; // "ZAP1"

/// Container for track metadata (ID3 tags).
#[derive(Serialize, Deserialize, Clone, Default, Debug)]
struct ZapMetadata {
    title: Option<String>,
    artist: Option<String>,
    album: Option<String>,
    year: Option<String>,
    genre: Option<String>,
    track_number: Option<u32>,
    comment: Option<String>,
    has_cover: bool,
}

/// Main file structure for the ZAP codec format.
#[derive(Serialize, Deserialize)]
struct ZapFile {
    /// File signature.
    magic: [u8; 4],
    /// Encoder version.
    version: String,
    /// Compressed audio payload (Zstd).
    data: Vec<u8>,
    /// Number of audio channels.
    channels: u16,
    /// Sample rate in Hz.
    rate: u32,
    /// Matrix shape of the spectral data (rows, cols).
    shape: (usize, usize),
    /// Global scaling factor derived during quantization.
    scale: f32,
    /// Track metadata.
    meta: ZapMetadata,
    /// Target fidelity score (informational).
    fidelity: f32,
    /// Total number of processing frames.
    num_frames: usize,
    /// DCT block size used.
    n_size: usize,
    /// Hop size (overlap) used.
    hop: usize,
    
    /// Psychoacoustic weights for Mid channel.
    weight_mid: Vec<f32>,
    /// Psychoacoustic weights for Side channel.
    weight_side: Vec<f32>,
    
    /// Coefficient used for pre-emphasis filter.
    pre_emphasis_coef: f32,
    /// Flag indicating if Mid/Side stereo coding was used.
    is_mid_side: bool,
    /// Exact number of samples in the original audio.
    original_len: usize,
}

// =============================================================================
//  DSP CORE (Digital Signal Processing)
// =============================================================================

/// Enhances transient response by emphasizing sudden positive amplitude changes.
/// This acts as a subtle transient shaper to preserve "punch" before compression.
fn apply_transient_shaper(samples: &mut [f32]) {
    for i in 1..samples.len() {
        let diff = (samples[i].abs() - samples[i - 1].abs()).max(0.0);
        // If a sharp transient is detected, slightly boost the sample
        if diff > 0.6 {
            samples[i] *= 1.0 + (diff * 0.05).min(0.005);
        }
    }
}

/// Applies a pre-emphasis filter (high-pass) to boost high frequencies before encoding.
/// This helps in preserving detail in the upper spectrum.
fn apply_pre_emphasis(samples: &mut [f32], channels: usize, coef: f32) {
    let frame_count = samples.len() / channels;
    for c in 0..channels {
        // Iterate backwards to avoid overwriting previous samples needed for calculation
        for i in (1..frame_count).rev() {
            let curr = i * channels + c;
            let prev = (i - 1) * channels + c;
            if curr < samples.len() && prev < samples.len() {
                samples[curr] = samples[curr] - coef * samples[prev];
            }
        }
    }
}

/// Reverses the pre-emphasis filter (de-emphasis) during decoding.
fn remove_pre_emphasis(samples: &mut [f32], channels: usize, coef: f32) {
    let frame_count = samples.len() / channels;
    for c in 0..channels {
        let mut prev = 0.0;
        for i in 0..frame_count {
            let idx = i * channels + c;
            if idx < samples.len() {
                samples[idx] = samples[idx] + coef * prev;
                prev = samples[idx];
            }
        }
    }
}

/// Converts Left/Right stereo data to Mid/Side representation.
/// Mid = (L+R)/2, Side = (L-R)/2.
fn convert_to_mid_side(samples: &mut [f32]) {
    for i in (0..samples.len()).step_by(2) {
        if i + 1 < samples.len() {
            let l = samples[i];
            let r = samples[i + 1];
            samples[i] = (l + r) * 0.5; // Mid
            samples[i + 1] = (l - r) * 0.5; // Side
        }
    }
}

/// Converts Mid/Side data back to Left/Right stereo.
fn convert_from_mid_side(samples: &mut [f32]) {
    for i in (0..samples.len()).step_by(2) {
        if i + 1 < samples.len() {
            let m = samples[i];
            let s = samples[i + 1];
            samples[i] = m + s; // Left
            samples[i + 1] = m - s; // Right
        }
    }
}

/// Analyzes the stereo field to determine if Mid/Side encoding is efficient.
/// Returns true if there is significant stereo information (Side energy).
fn should_use_midside(samples: &[f32]) -> bool {
    let mut mid_energy = 0.0;
    let mut side_energy = 0.0;
    let step = 200; // Check every 200th sample for performance
    
    for i in (0..samples.len()).step_by(step) {
        if i + 1 >= samples.len() { break; }
        let l = samples[i];
        let r = samples[i + 1];
        let m = (l + r) * 0.5;
        let s = (l - r) * 0.5;
        mid_energy += m.abs();
        side_energy += s.abs();
    }
    
    // If side energy is very low (< 5% of mid), standard LR might be safer/simpler,
    // though here we return false (meaning "don't force MS" or "not worth it").
    if side_energy < mid_energy * 0.05 { return false; }
    true
}

/// Applies a DC Blocker filter to remove constant offsets (0Hz) from the signal.
fn apply_dc_blocker(pcm: &mut [f32], channels: usize) {
    let r = 0.9995;
    for c in 0..channels {
        let mut x_prev = 0.0;
        let mut y_prev = 0.0;
        for i in (c..pcm.len()).step_by(channels) {
            let x = pcm[i];
            let y = x - x_prev + r * y_prev;
            pcm[i] = y;
            x_prev = x;
            y_prev = y;
        }
    }
}

/// cleans the frequency spectrum by removing sub-bass, ultrasonic frequencies,
/// and applying a noise gate to negligible values.
fn scrub_spectrum(frame: &mut [f32], n_size: usize, rate: u32) {
    let bin_hz = rate as f32 / 2.0 / n_size as f32;
    for i in 0..n_size.min(frame.len()) {
        let freq = i as f32 * bin_hz;
        
        // Hard cut below 10Hz and above 22kHz
        if freq < 10.0 { frame[i] = 0.0; continue; }
        if freq > 22000.0 { frame[i] = 0.0; continue; }
        
        // Noise Gate / Denoise
        let val = frame[i].abs();
        if val < 0.00003 {
             let ratio = val / 0.00003;
             frame[i] *= ratio * ratio; // Soft knee attenuation
        }
    }
}

/// Generates psychoacoustic weighting curves.
/// Boosts sensitivity in the "presence" range (2k-5kHz).
fn get_studio_weights(n_size: usize, rate: u32, is_side: bool) -> Vec<f32> {
    let mut weights = vec![1.0; n_size];
    for i in 0..n_size {
        let freq = i as f32 * (rate as f32 / 2.0) / n_size as f32;
        let mut w = 1.0;
        
        // Slight boost in human hearing sensitive range
        if freq >= 2000.0 && freq < 5000.0 { w = 1.1; }
        
        // Lower weight for side channel (less perceptible)
        if is_side { w *= 0.95; }
        
        weights[i] = w;
    }
    weights
}

/// Non-linear quantization using a power law (Gamma correction style).
/// Compresses dynamic range into integer space.
fn non_linear_quantize(val: f32) -> f32 {
    if val.abs() < LINEAR_DEADZONE { return 0.0; }
    val.signum() * val.abs().powf(POWER_LAW_EXPONENT)
}

/// Reverses the non-linear quantization.
fn non_linear_dequantize(val: f32) -> f32 {
    if !val.is_finite() { return 0.0; }
    val.signum() * val.abs().powf(INV_POWER_LAW)
}

// =============================================================================
//  UTILITIES / UI
// =============================================================================

fn print_banner() {
    println!("");
    println!("╔═══════════════════════════════════════════════════════════╗");
    println!("║                   ZAP Audio Codec v1.0.0                  ║");
    println!("║                Zero Audible Psychoacoustic                ║");
    println!("╚═══════════════════════════════════════════════════════════╝");
    println!("");
}

fn format_filesize(size: u64) -> String {
    let units = ["B", "KB", "MB", "GB"];
    let mut s = size as f64;
    for unit in units {
        if s < 1024.0 { return format!("{:.2} {}", s, unit); }
        s /= 1024.0;
    }
    format!("{:.2} TB", s)
}

fn format_time(seconds: u64) -> String {
    let m = seconds / 60;
    let s = seconds % 60;
    format!("{:02}:{:02}", m, s)
}

/// Prints a dynamic progress bar to the console.
fn print_status_line(action: &str, current: usize, total: usize, start_time: Option<Instant>) {
    let width: usize = 30;
    let percent = if total > 0 { (current as f64 / total as f64) * 100.0 } else { 0.0 };
    let filled = (percent / 100.0 * width as f64) as usize;
    let bar = format!("{}{}", "█".repeat(filled), "░".repeat(width.saturating_sub(filled)));
    
    let time_info = if let Some(start) = start_time {
        let elapsed = start.elapsed().as_secs_f64();
        let speed = if elapsed > 0.0 { current as f64 / elapsed } else { 0.0 };
        let remaining = if speed > 0.0 { (total - current) as f64 / speed } else { 0.0 };
        format!("ETA {:.0}s", remaining)
    } else { String::new() };
    
    print!("\r  {:<10} ▐{}▌ {:>5.1}%  {}", action, bar, percent, time_info);
    std::io::stdout().flush().unwrap();
}

// =============================================================================
//  ENCODER
// =============================================================================

fn process_audio(path: &str, manual_meta: Option<ZapMetadata>) {
    let _start_time = Instant::now();
    let p = Path::new(path);
    if !p.exists() { println!("✗ File not found"); return; }

    // --- Format Validation (Lossy Lockout) ---
    // ZAP enforces lossless sources to ensure studio quality input.
    if let Some(ext) = p.extension() {
        let ext_str = ext.to_string_lossy().to_lowercase();
        match ext_str.as_str() {
            "wav" | "flac" | "aiff" | "aif" => {}, // OK
            "mp3" | "ogg" | "m4a" | "aac" | "wma" | "opus" => {
                println!("\n⛔ ERROR: Lossy format detected (.{})!", ext_str);
                println!("   ZAP is designed for lossless sources only.");
                println!("   Please use WAV, FLAC or AIFF inputs.");
                return;
            },
            _ => { println!("⚠️ Warning: Unknown format."); }
        }
    }

    let src = File::open(path).expect("File error");
    let file_size = src.metadata().unwrap().len();
    print_banner();
    println!("📂 Input: {} ({})", p.file_name().unwrap().to_string_lossy(), format_filesize(file_size));
    
    // --- Audio Decoding (using Symphonia) ---
    let mss = MediaSourceStream::new(Box::new(src), Default::default());
    let mut hint = Hint::new();
    if let Some(ext) = p.extension() { hint.with_extension(&ext.to_string_lossy()); }
    
    let probed = match symphonia::default::get_probe().format(&hint, mss, &Default::default(), &Default::default()) {
        Ok(p) => p, Err(_) => { println!("✗ Format Error"); return; }
    };
    let mut decoder = symphonia::default::get_codecs().make(&probed.format.default_track().unwrap().codec_params, &Default::default()).unwrap();

    let mut samples = Vec::new();
    let mut rate = 0;
    let mut channels = 0;
    let mut format = probed.format;

    print!(" ➔ Decoding PCM...");
    loop {
        match format.next_packet() {
            Ok(packet) => {
                let decoded = decoder.decode(&packet).unwrap();
                let spec = *decoded.spec();
                rate = spec.rate;
                channels = spec.channels.count();
                // Interleave samples into a single buffer
                let mut buf = symphonia::core::audio::SampleBuffer::<f32>::new(decoded.capacity() as u64, spec);
                buf.copy_interleaved_ref(decoded);
                samples.extend_from_slice(buf.samples());
            }
            Err(_) => break,
        }
    }
    println!(" Done. ({}Hz, {}ch)", rate, channels);
    if samples.is_empty() { println!("No samples."); return; }

    // --- Metadata Collection ---
    let mut meta = ZapMetadata::default();
    if let Ok(tagged) = Probe::open(path).and_then(|p| p.read()) {
        if let Some(tag) = tagged.primary_tag() {
            if let Some(v) = tag.title() { meta.title = Some(v.to_string()); }
            if let Some(v) = tag.artist() { meta.artist = Some(v.to_string()); }
            if let Some(v) = tag.album() { meta.album = Some(v.to_string()); }
            if let Some(v) = tag.year() { meta.year = Some(v.to_string()); }
            if let Some(v) = tag.genre() { meta.genre = Some(v.to_string()); }
            if let Some(v) = tag.track() { meta.track_number = Some(v); }
            if !tag.pictures().is_empty() { meta.has_cover = true; }
        }
    }
    // Override with CLI arguments if provided
    if let Some(m) = manual_meta {
        if let Some(t) = m.title { meta.title = Some(t); }
        if let Some(a) = m.artist { meta.artist = Some(a); }
        if let Some(al) = m.album { meta.album = Some(al); }
    }

    // --- Padding & Buffer Preparation ---
    let n_size = BLOCK_SIZE;
    let hop = n_size / 2;
    let channels_usize = channels as usize;
    let original_len = samples.len(); 

    // Pad front (priming the filter)
    let pad_front_frames = hop;
    let mut padded_samples = Vec::with_capacity(original_len + hop * 4 * channels_usize);
    for i in (1..=pad_front_frames).rev() {
        let idx = i * channels_usize;
        if idx < samples.len() {
            for c in 0..channels_usize {
                if idx + c < samples.len() { padded_samples.push(samples[idx + c]); } else { padded_samples.push(0.0); }
            }
        }
    }
    padded_samples.extend_from_slice(&samples);

    // Pad back to match block alignment
    let current_len = padded_samples.len();
    let remainder = (current_len / channels_usize) % hop;
    let pad_back_frames = if remainder > 0 { hop - remainder } else { 0 } + hop * 2;
    padded_samples.resize(current_len + pad_back_frames * channels_usize, 0.0);

    let mut samples = padded_samples;
    let duration = original_len as f64 / (rate as f64 * channels as f64);

    // Apply Filters
    apply_pre_emphasis(&mut samples, channels, 0.28);
    apply_transient_shaper(&mut samples);

    let use_mid_side = if channels == 2 { should_use_midside(&samples) } else { false };
    if use_mid_side { convert_to_mid_side(&mut samples); }

    let num_frames = (samples.len() / channels_usize).saturating_sub(n_size) / hop;

    // Window Function (Sine Window for perfect MDCT/DCT4 overlap-add)
    let window: Vec<f32> = (0..n_size)
        .map(|i| ((PI / n_size as f64) * (i as f64 + 0.5)).sin() as f32)
        .collect();
    
    // De-interleave data for processing
    let mut channel_data = vec![vec![0.0; samples.len()/channels_usize]; channels_usize];
    for (i, s) in samples.iter().enumerate() { channel_data[i%channels_usize][i/channels_usize] = *s; }

    let mut planner = DctPlanner::new();
    let dct = Arc::new(planner.plan_dct4(n_size)); 
    let norm = (2.0 / n_size as f64).sqrt() as f32;

    // --- Spectral Analysis (Parallel) ---
    let analysis_start = Instant::now();
    let all_spectra: Vec<Vec<f32>> = (0..num_frames).into_par_iter().map(|f| {
        let dct_inner = dct.clone(); 
        let mut frames_for_step = Vec::with_capacity(channels_usize);

        for c in 0..channels_usize {
            let mut frame = vec![0.0; n_size];
            let base_idx = f * hop;
            for i in 0..n_size {
                if base_idx + i < channel_data[c].len() {
                    frame[i] = channel_data[c][base_idx + i] * window[i];
                }
            }
            dct_inner.process_dct4(&mut frame);
            for x in &mut frame { *x *= norm; } // Normalize
            
            scrub_spectrum(&mut frame, n_size, rate);
            frames_for_step.push(frame);
        }
        frames_for_step.into_iter().flatten().collect()
    }).collect();
    
    print_status_line("Analyzing", num_frames, num_frames, Some(analysis_start));
    println!("");

    let flat_spectra: Vec<f32> = all_spectra.into_iter().flatten().collect();
    
    // Calculate global weights
    let w_mid = get_studio_weights(n_size, rate, false);
    let w_side = get_studio_weights(n_size, rate, true);
    
    let mut full_weights = Vec::with_capacity(flat_spectra.len());
    for _ in 0..num_frames {
        for c in 0..channels_usize {
            if use_mid_side && c == 1 { full_weights.extend_from_slice(&w_side); }
            else { full_weights.extend_from_slice(&w_mid); }
        }
    }

    // --- Adaptive Quantization Optimization Loop ---
    let target_bytes = (MAX_ALLOWED_KBPS as f64 * 1000.0 / 8.0) * duration;
    let mut div = 10000.0; // Initial scaling factor
    
    let mut best_data = Vec::new();
    let mut best_size = f64::MAX;
    let compress_start = Instant::now();

    let stride = n_size * channels_usize;

    // Attempt to find the optimal scalar to fit the target bitrate
    for iter in 0..30 {
        let mut delta_buf = Vec::with_capacity(flat_spectra.len());
        let mut quant_history = vec![0i32; flat_spectra.len()]; 

        for (i, &val) in flat_spectra.iter().enumerate() {
            // Apply weight and scalar
            let s = (val * PRECISION_BASE * full_weights[i]) / div;
            
            let q_float = non_linear_quantize(s);
            let mut q_int = if q_float.is_finite() { q_float.round() as i32 } else { 0 };
            q_int = q_int.clamp(-2147483640, 2147483640);
            
            quant_history[i] = q_int;

            // Delta encoding: Only store difference from the previous corresponding bin
            let pred = if i >= stride { quant_history[i - stride] } else { 0 };
            let delta = q_int.wrapping_sub(pred);
            delta_buf.push(delta);
        }

        let mut raw = Vec::with_capacity(delta_buf.len() * 4);
        for d in &delta_buf { raw.extend_from_slice(&d.to_le_bytes()); }
        
        // Fast test compression
        let comp = zstd::stream::encode_all(&raw[..], 3).unwrap();
        let size = comp.len() as f64;
        let valid = size <= target_bytes;
        
        // Keep result if it's the best valid one found, or if we haven't found a valid one yet
        if (valid && size > best_size && best_size < target_bytes) || (!valid && size < best_size) || (valid && best_size > target_bytes) {
             if valid { best_size = size; best_data = delta_buf.clone(); }
             else if best_data.is_empty() || size < best_size { best_size = size; best_data = delta_buf.clone(); }
        }
        
        print_status_line("Optimizing", iter + 1, 30, Some(compress_start));
        
        // Adjust scalar for next iteration
        if valid { 
            div *= 0.94; // Reduce scalar (increase quality)
        } else { 
            let ratio = size / target_bytes;
            div *= ratio.powf(1.5) as f32; // Increase scalar (decrease quality)
        }
        
        // If we are extremely close to the limit, stop early
        if valid && (target_bytes - size) < target_bytes * 0.01 { 
            print_status_line("Optimizing", 30, 30, Some(compress_start));
            break; 
        }
    }
    println!("");

    let mut raw = Vec::with_capacity(best_data.len()*4);
    for d in &best_data { raw.extend_from_slice(&d.to_le_bytes()); }
    
    print!(" ➔ Final Packing (Zstd Ultra 21)...");
    let final_blob = zstd::stream::encode_all(&raw[..], 21).unwrap();
    println!(" Done.");

    // --- File Construction ---
    let zfile = ZapFile {
        magic: ZAP_MAGIC,
        version: "1.0.0".into(), 
        data: final_blob,
        channels: channels as u16,
        rate,
        shape: (num_frames * channels_usize, n_size),
        scale: div,
        meta: meta.clone(),
        fidelity: 99.9,
        num_frames,
        n_size,
        hop,
        
        weight_mid: w_mid,
        weight_side: w_side,
        
        pre_emphasis_coef: 0.28,
        is_mid_side: use_mid_side, 
        original_len: original_len, 
    };

    let out_path = p.with_extension("zap");
    let f = File::create(&out_path).unwrap();
    bincode::serialize_into(f, &zfile).unwrap();
    let fs = std::fs::metadata(&out_path).unwrap().len();
    
    println!("\n{}", "═".repeat(60));
    println!("✓ ENCODING SUCCESSFUL (v1.0.0)");
    println!("  File:    {}", out_path.display());
    println!("  Size:    {}", format_filesize(fs));
    println!("{}\n", "═".repeat(60));
}

// =============================================================================
//  PLAYER / DECODER
// =============================================================================

fn play_or_decode(path: &str, decode: bool, out: Option<String>) {
    let p = Path::new(path);
    if !p.exists() { println!("File not found"); return; }
    
    print_banner();
    if decode { println!("📂 DECODING..."); } 
    else { println!("📂 PLAYING..."); }

    let f = File::open(path).unwrap();
    let mut reader = std::io::BufReader::with_capacity(1024*1024, f);
    
    let zap: ZapFile = match bincode::deserialize_from(&mut reader) {
        Ok(v) => v, Err(_) => { println!("Invalid File"); return; }
    };

    println!("  Title:  {}", zap.meta.title.clone().unwrap_or("?".into()));
    println!("  Artist: {}", zap.meta.artist.clone().unwrap_or("?".into()));

    print!(" ➔ Buffering...");
    let raw = zstd::stream::decode_all(&zap.data[..]).unwrap();
    println!(" OK");

    let mut q_vals = Vec::with_capacity(raw.len()/4);
    for chunk in raw.chunks_exact(4) {
        q_vals.push(i32::from_le_bytes([chunk[0],chunk[1],chunk[2],chunk[3]]));
    }

    // --- Reconstruction (Delta Decoding) ---
    let rows = zap.shape.0; 
    let cols = zap.shape.1; 
    let stride_idx = zap.n_size * zap.channels as usize;

    let mut reconstructed_history = vec![0i32; q_vals.len()];
    
    for i in 0..q_vals.len() {
        let delta = q_vals[i];
        let pred = if i >= stride_idx {
            reconstructed_history[i - stride_idx]
        } else {
            0
        };
        reconstructed_history[i] = pred.wrapping_add(delta);
    }
    
    let w_mid = zap.weight_mid.clone();
    let w_side = zap.weight_side.clone();
    
    let mut spectra = Vec::with_capacity(rows);
    let mut ptr = 0;
    
    // Dequantize
    for r in 0..rows { 
        let mut row = Vec::with_capacity(cols);
        let is_side_ch = zap.is_mid_side && (r % 2 == 1);
        let active_w = if is_side_ch { &w_side } else { &w_mid };

        for c in 0..cols {
            let q_int = reconstructed_history[ptr];
            ptr += 1;
            
            let w = if c < active_w.len() { active_w[c] } else { 1.0 };
            let val = (non_linear_dequantize(q_int as f32) * zap.scale) / (PRECISION_BASE * w);
            row.push(val);
        }
        spectra.push(row);
    }

    // --- Synthesis (Inverse DCT) ---
    print!(" ➔ Synthesis (Multi-Core)...");
    let n = zap.n_size;
    let hop = zap.hop;
    let channels = zap.channels as usize;
    
    let mut planner = DctPlanner::new();
    let dct = Arc::new(planner.plan_dct4(n));
    let norm = (2.0 / n as f64).sqrt() as f32;
    let win: Vec<f32> = (0..n).map(|i| ((PI / n as f64) * (i as f64 + 0.5)).sin() as f32).collect();

    let chunks: Vec<Vec<f32>> = spectra.par_iter().map(|s| {
        let dct_inner = dct.clone();
        let mut frame = s.clone();
        
        dct_inner.process_dct4(&mut frame);
        for i in 0..n { frame[i] *= norm * win[i]; }
        frame
    }).collect();
    println!(" OK");

    // Overlap-Add method to reconstruct PCM
    let output_len = (zap.num_frames + 1) * hop;
    let mut pcm = vec![0.0; output_len * channels];
    
    for (f, chunk) in chunks.iter().enumerate() {
        let c = f % channels;
        let base = (f / channels) * hop;
        for i in 0..n {
            let out_idx = (base + i) * channels + c;
            if out_idx < pcm.len() { pcm[out_idx] += chunk[i]; }
        }
    }

    // --- Post Processing ---
    if zap.is_mid_side { convert_from_mid_side(&mut pcm); }
    
    remove_pre_emphasis(&mut pcm, channels, zap.pre_emphasis_coef);
    apply_dc_blocker(&mut pcm, channels);
    
    // Hard Limiter (Safety)
    for x in &mut pcm { 
        if *x > 1.0 { *x = 1.0; }
        else if *x < -1.0 { *x = -1.0; }
    } 

    // Remove Padding
    let pad_front_samples = hop * channels;
    if pcm.len() > pad_front_samples {
        pcm.drain(0..pad_front_samples);
    }

    if zap.original_len > 0 && pcm.len() > zap.original_len {
        pcm.truncate(zap.original_len);
    }

    if decode {
        let name = out.unwrap_or("decoded.wav".into());
        let spec = hound::WavSpec { channels: zap.channels, sample_rate: zap.rate, bits_per_sample: 16, sample_format: hound::SampleFormat::Int };
        let mut w = hound::WavWriter::create(&name, spec).unwrap();
        for s in pcm { w.write_sample((s * 32700.0) as i16).unwrap(); }
        println!("✓ Saved WAV: {}", name);
    } else {
        play_pcm(pcm, zap.channels, zap.rate);
    }
}

/// Plays the PCM data using the default system output device.
fn play_pcm(pcm: Vec<f32>, channels: u16, rate: u32) {
    let host = cpal::default_host();
    let dev = host.default_output_device().unwrap();
    let cfg = dev.default_output_config().unwrap();
    let dev_rate = cfg.sample_rate().0;
    
    // Simple Linear Resampler if system rate differs from file rate
    let final_pcm = if rate != dev_rate {
        let ratio = rate as f32 / dev_rate as f32;
        let target_len = (pcm.len() as f32 / ratio) as usize;
        let mut res = Vec::with_capacity(target_len);
        
        for i in 0..target_len/channels as usize {
             let pos = i as f32 * ratio;
             let idx_floor = pos.floor() as usize;
             let frac = pos - idx_floor as f32;
             
             for c in 0..channels as usize {
                 let idx_a = idx_floor * channels as usize + c;
                 let idx_b = (idx_floor + 1) * channels as usize + c;
                 
                 let val_a = if idx_a < pcm.len() { pcm[idx_a] } else { 0.0 };
                 let val_b = if idx_b < pcm.len() { pcm[idx_b] } else { 0.0 };
                 
                 let sample = val_a * (1.0 - frac) + val_b * frac;
                 res.push(sample);
             }
        }
        res
    } else { pcm };

    let buf = Arc::new(Mutex::new(final_pcm));
    let pos = Arc::new(Mutex::new(0usize));
    let b = buf.clone();
    let p = pos.clone();
    
    let stream = dev.build_output_stream(&cfg.config(), move |data: &mut [f32], _| {
        let b = b.lock().unwrap();
        let mut p = p.lock().unwrap();
        for s in data.iter_mut() {
            if *p < b.len() { *s = b[*p]; *p += 1; } else { *s = 0.0; }
        }
    }, move |_| {}, None).unwrap();
    
    stream.play().unwrap();
    let run = Arc::new(AtomicBool::new(true));
    let r = run.clone();
    ctrlc::set_handler(move || r.store(false, Ordering::SeqCst)).ok();
    
    println!("{}", "═".repeat(60));
    println!("▶ PLAYBACK (Ctrl+C to stop)");
    println!("{}", "═".repeat(60));

    while run.load(Ordering::SeqCst) {
        let p = *pos.lock().unwrap();
        let len = buf.lock().unwrap().len();
        if p >= len { break; }
        let sec = p as u64 / channels as u64 / dev_rate as u64;
        let tot = len as u64 / channels as u64 / dev_rate as u64;
        
        print!("\r  🔊 [{}/{}] ", format_time(sec), format_time(tot));
        std::io::stdout().flush().ok();
        std::thread::sleep(Duration::from_millis(100));
    }
    println!("\nStopped.");
}

// =============================================================================
//  MAIN
// =============================================================================

fn main() {
    let args: Vec<String> = env::args().collect();
    if args.len() < 2 { print_help(); return; }
    match args[1].as_str() {
        "encode" => {
            let mut manual_meta = ZapMetadata::default();
            let mut i = 3;
            while i < args.len() {
                match args[i].as_str() {
                    "--title"   => if i+1 < args.len() { manual_meta.title = Some(args[i+1].clone()); i+=1; },
                    "--artist"  => if i+1 < args.len() { manual_meta.artist = Some(args[i+1].clone()); i+=1; },
                    "--album"   => if i+1 < args.len() { manual_meta.album = Some(args[i+1].clone()); i+=1; },
                    "--year"    => if i+1 < args.len() { manual_meta.year = Some(args[i+1].clone()); i+=1; },
                    "--genre"   => if i+1 < args.len() { manual_meta.genre = Some(args[i+1].clone()); i+=1; },
                    "--comment" => if i+1 < args.len() { manual_meta.comment = Some(args[i+1].clone()); i+=1; },
                    "--track"   => if i+1 < args.len() { 
                        if let Ok(n) = args[i+1].parse::<u32>() { manual_meta.track_number = Some(n); }
                        i+=1; 
                    },
                    _ => {}
                }
                i += 1;
            }
            if args.len() >= 3 {
               process_audio(&args[2], Some(manual_meta));
            } else {
               println!("Error: Input file missing.");
            }
        },
        "play" => {
            if args.len() < 3 { println!("Input needed"); return; }
            play_or_decode(&args[2], false, None);
        },
        "decode" => {
            if args.len() < 3 { println!("Input needed"); return; }
            let out = if args.len() > 3 { Some(args[3].clone()) } else { None };
            play_or_decode(&args[2], true, out);
        },
        _ => print_help(),
    }
}

fn print_help() {
    println!("USAGE:");
    println!("  ./zap_codec encode <input.wav/flac> [TAGS]");
    println!("  ./zap_codec play <file.zap>");
    println!("  ./zap_codec decode <file.zap> [out.wav]");
}